//==========================================================================
// Game.cpp
//  This class creates a simulated role playing game to demonstrate the
//  Memento Design Pattern.  At regular intervals a GameMemento object
//  is updated and saved to disk to preserve the last state of the game.
//  Author Dr. Rick Coleman 
//==========================================================================
#pragma warning(disable:4996)

#include "Game.h"
#include "GameMemento.h"
#include <sys/types.h>     // needed for the _ftime() function
#include <sys/timeb.h>     // contains definition of _timeb struct
#include <time.h>          // needed for other time functions and structures
#include <iostream>
#include <fstream>
#include <math.h>
#include <windows.h>  // for the Sleep() function

using namespace std;

//---------------------------------
// Default constructor 
//---------------------------------
Game::Game()
{
	m_GameMemento = new GameMemento(); // Create GameMemento in which to save the game state
}

//---------------------------------
// Destructor 
//---------------------------------
Game::~Game()
{
	delete m_GameMemento;
}

//---------------------------------
// Main game playing function.
//---------------------------------
void Game::PlayGame()
{
	int minCounter = 1;			// Minute time for game
	int playerX = BOARDLEFT;	// Starting location of the player
	int playerY = BOARDTOP;
	int xInc = 3;
	int yInc = 1;
	double curTime, nextTime;
    struct _timeb   tStruct;
	char recoveryFile[64] = "";
	
	// Init the random number generator
	srand((unsigned int)(time(NULL)));
	// Create a GameMemento
	m_GameMemento = new GameMemento();
	// Get the name used for recovery files
	strcpy(recoveryFile, m_GameMemento->GetRecoveryFileName());

	// Check to see if there is a recovery file
    ifstream ifstr;
	ifstr.open(recoveryFile, ifstream::in); 
	if(ifstr.good())
	{
		ifstr.close();  // Close the file.  InitGame will reopen it
		cout << "\n\nA crash recovery file has been found.\n";
		cout << "Do you want to load it and resume playing the game?\n";
		cout << "Type Y or N then press Enter: ";
		char ch;
		cin.get(ch);
		if((ch == 'Y') || (ch == 'y'))
		{
			// If there is a recovery file call InitGame passing it the file name
			minCounter = InitGame(recoveryFile);
		}
		else
		{
			// Delete the file
			// No backup file found or user doesn't want to use it so call 
			//   InitGame to set the starting state of a new game
			minCounter = InitGame(NULL);
		}
    }
	else 
	{
		// No backup file found so call InitGame to set the starting state of a new game
		minCounter = InitGame(NULL);
	}
	
	// Enter the game loop to play the game
	boolean gameOver = false;
    _ftime(&tStruct);	// Get start time
    curTime = tStruct.time + (((double)(tStruct.millitm)) / 1000.0); // Convert to double
	double timeInc = 1.0;  // Update at 1 seond intervals
	nextTime = curTime + timeInc;
	if(minCounter > 1) // We are recovering
	{
		for(int i=0; i<minCounter; i++)
		{
			// Reset player position
			playerX+=xInc;
			playerY+=yInc;
			// If at the bounds of the area reverse directions
			if((playerX <= BOARDLEFT) || (playerX >= BOARDRIGHT))
			{
				xInc*=(-1);
				if(playerX < BOARDLEFT) playerX = BOARDLEFT;
				if(playerX >= BOARDRIGHT) playerX = (BOARDRIGHT-1);
			}
			if((playerY <= BOARDTOP) || (playerY >= BOARDBOTTOM))
			{
				yInc*=(-1);
				if(playerY < BOARDTOP) playerY = BOARDTOP;
				if(playerY > BOARDBOTTOM) playerY = BOARDBOTTOM;
			}
		}
	}
	while(!gameOver)
	{
		// Play the game here
		_ftime(&tStruct);	// Get start time
		curTime = tStruct.time + (((double)(tStruct.millitm)) / 1000.0); // Convert to double
		if(curTime >= nextTime)
		{
			nextTime = curTime + timeInc;
			// redraw the player
			playerX+=xInc;
			playerY+=yInc;
			// If at the bounds of the area reverse directions
			if((playerX <= BOARDLEFT) || (playerX >= BOARDRIGHT))
			{
				xInc*=(-1);
				if(playerX < BOARDLEFT) playerX = BOARDLEFT;
				if(playerX >= BOARDRIGHT) playerX = (BOARDRIGHT-1);
			}
			if((playerY <= BOARDTOP) || (playerY >= BOARDBOTTOM))
			{
				yInc*=(-1);
				if(playerY <= BOARDTOP) playerY = BOARDTOP;
				if(playerY >= BOARDBOTTOM) playerY = BOARDBOTTOM;
			}
			PrintGameBoard(playerX, playerY);
			cout << "You have been playing for " << minCounter << " minutes.\n";
			minCounter++;
			SaveGameState(minCounter);
			// Roll 1D100
			int roll = rand() % 100;
			if(roll <= 5)		// 1 in 20 change of a disaster
			{
				roll = rand() % 10;
				if(roll < 5)
				{
					Sleep(2000);
					cout << "\n\nGasp! Your 16th level warrior has been destroyed.\n";
					cout << "Game Over Man!!!\n\n";
					Sleep(2000);
					return;
				}
				else
				{
					Sleep(2000);
					cout << "\n\nImminent program crash detected.\n";
					Sleep(2000);
					return;
				}
			}
		}
	}
}
	
//---------------------------------
// Print the game board with player
//   position.
//---------------------------------
void Game::PrintGameBoard(int X, int Y)
{
	static int lastX = BOARDLEFT;
	static int lastY = BOARDTOP;

	// Print a top border
	for(int i=0; i<BOARDWIDTH; i++)
		cout << "=";
	cout << endl;
	// Clear the last player position
	m_GameBoard[lastY][lastX] = '.';
	// Set this player position
	m_GameBoard[Y][X] = 'X';
	// Print the board
	for(int i=0; i<BOARDHEIGHT; i++)
	{
		cout << m_GameBoard[i] << endl;
	}
	// Print a bottom border
	for(int i=0; i<BOARDWIDTH; i++)
		cout << "=";
	cout << endl;
	// Save this XY for next time
	lastX = X;
	lastY = Y;
}

	
//---------------------------------
// Save the game state by writing
//  the GameMemento to disk. 
//---------------------------------
void Game::SaveGameState(int state)
{
	m_GameMemento->SetStateFlag(state);
	m_GameMemento->UpdateGameMemento();
}
	
//---------------------------------
/** Initialize the game. */
//---------------------------------
int Game::InitGame(char *recoveryFile)
{
	// Initialize the game board
	for(int i=0; i<BOARDHEIGHT; i++)
	{
		for(int j=0; j<BOARDWIDTH; j++) // Fill row with dots
			m_GameBoard[i][j] = '.';
		m_GameBoard[i][BOARDRIGHT] = '\0'; // Put null terminator at end
	}
	int minCount = 1; // In case we read another from the file
	
	if(recoveryFile == NULL)
	{
		// Set up a new game
		m_GameMemento->SetPlayerData("This string represents the data defining the player character.");
		m_GameMemento->SetCreatureData("This string represents the data defining all creatures in the game.");
		m_GameMemento->SetTreasureData("This string represents the data defining all the treasure items in the game.");
		m_GameMemento->SetGameScenarioData("This string represents the data defining the game scenario.");
		m_GameMemento->SetPlayerStateData("This string represents the player's current state in the game.");
		m_GameMemento->SetStateFlag(1);
	}
	else
	{
		ifstream ifstr;
		ifstr.open(recoveryFile, ifstream::in); 
		if(ifstr.good())
		{
			char line[128];  // Char arry to store lines from file into

			getNextLine(ifstr, line, 127); // Read and skip the opening comment
			getNextLine(ifstr, line, 127); // Read the player character data line
			m_GameMemento->SetPlayerData(line);
			getNextLine(ifstr, line, 127); // Read the creature data line
			m_GameMemento->SetCreatureData(line);
			getNextLine(ifstr, line, 127); // Read the treasure data line
			m_GameMemento->SetTreasureData(line);
			getNextLine(ifstr, line, 127); // Read the game scenario data line
			m_GameMemento->SetGameScenarioData(line);
			getNextLine(ifstr, line, 127); // Read the player state data line
			m_GameMemento->SetPlayerStateData(line);
			getNextLine(ifstr, line, 127); // Read the state flag data line
			char *cptr = strstr(line, "= "); // find the number string
			// Find the end of the number
			char *cpt2 = cptr;
			// Increment to start of number
			cpt2++;	// Skip the '='
			cpt2++; // Skip the ' '
			minCount = atoi(cpt2); 
			m_GameMemento->SetStateFlag(minCount);
		}
		else
		{
			// Report an error
			cout << "\n\nError encountered reading the recovery file.  Setting new game.\n";
			Sleep(3000);
			// Restart new game
			InitGame(NULL);
		}			
	}
	return minCount;
}

//--------------------------------------------
// getNextLine_CPP() 
// Read a line from a data file.
//--------------------------------------------
void Game::getNextLine(ifstream& inFile, char *line, int lineLen)
{
    int    done = false;
    while(!done)
    {
        inFile.getline(line, 128);
        
        if(inFile.good())    // If a line was successfully read
        {
            if(strlen(line) == 0)  // Skip any blank lines
                continue;
            else if(line[0] == '#')  // Skip any comment lines
                continue;
            else done = true;    // Got a valid data line so return with this line
        }
        else
        {
            strcpy(line, "");
            return;
        }
    } // end while
}

